/*
   Arduino Aquarium Controller (well, the start of one anyway)
   This uses bits and pieces of code from all over the internet.
   These are some of the websites that were used:
      - http://milesburton.com/Dallas_Temperature_Control_Library
      - http://code.google.com/p/sdfatlib/
      - http://www.instructables.com/id/ServDuino-Arduino-Webserver/
      - http://www.ladyada.net/learn/arduino/ethfiles.html
      
   Andre Miller 2012-03-08 (arduino@andremiller.net)      
*/

#include <SPI.h>
#include <SdFat.h>
#include <SdFatUtil.h>
#include <Ethernet.h>

#include <OneWire.h>
#include <DallasTemperature.h>

// Data wire for temperature sensor is plugged into port 22 on the Arduino
#define ONE_WIRE_BUS 22
#define TEMPERATURE_PRECISION 12

const char *menuItems[][2] = {
  { "Home", "/" },
  { "Scan", "tempinit" },
  { "Temperature", "temp"},
  { "Analog", "analog" }
};

int numberOfMenuItems = 4;

// Setup a oneWire instance to communicate with any OneWire devices (not just Maxim/Dallas temperature ICs)
OneWire oneWire(ONE_WIRE_BUS);

// Pass our oneWire reference to Dallas Temperature. 
DallasTemperature sensors(&oneWire);

int numberOfDevices; // Number of temperature devices found

DeviceAddress tempDeviceAddress; // We'll use this variable to store a found device address


uint8_t bufindex;  //Buffer index, used to keep track of current buffer byte
const uint8_t maxbyte=255;  //The maximum allowable buffer length
uint8_t buf[maxbyte];  //Creates the buffer with a length equal to the max length

byte mac[] = {
  0xBA,0xBE,192,168,2,92};
byte ip[] = {
  192,168,2,92};
char rootFileName[] = "index.htm";
char rootWebDir[] = "www"; // Subdirectory on SD card that contains the website
EthernetServer server(8081);

Sd2Card card;
SdVolume volume;
SdFile root;
SdFile file;

#define error(s) error_P(PSTR(s))
void error_P(const char* str) {
  PgmPrint("error: ");
  SerialPrintln_P(str);
  if (card.errorCode()) {
    PgmPrint("SD error: ");
    Serial.print(card.errorCode(), HEX);
    Serial.print(',');
    Serial.println(card.errorData(), HEX);
  }
  while(1);
}

void setup() {
  Serial.begin(9600);
  PgmPrint("Free RAM: ");
  Serial.println(FreeRam());
  pinMode(10, OUTPUT);
  digitalWrite(10, HIGH);
  if (!card.init(SPI_FULL_SPEED, 4)) error("card.init failed!");
  if (!volume.init(&card)) error("vol.init failed!");
  PgmPrint("Volume is FAT");
  Serial.println(volume.fatType(),DEC);
  Serial.println();
  if (!root.openRoot(&volume)) error("openRoot failed");
  PgmPrintln("Files found in root:");
  root.ls(LS_DATE | LS_SIZE);
  Serial.println();
  PgmPrintln("Files found in all dirs:");
  root.ls(LS_R);
  Serial.println();
  PgmPrintln("Done");
  Ethernet.begin(mac, ip);
  server.begin();
  Serial.println("setup() done");
}
#define BUFSIZ 100

void loop()
{
  char clientline[BUFSIZ];
  char *filename;
  char webfilename[50];
  int index = 0;
  int image = 0;
  EthernetClient client = server.available();
  if (client) {
    boolean current_line_is_blank = true;
    index = 0;
    while (client.connected()) {
      if (client.available()) {
        char c = client.read();
        if (c != '\n' && c != '\r') {
          clientline[index] = c;
          index++;
          if (index >= BUFSIZ)
            index = BUFSIZ -1;
          continue;
        }
        clientline[index] = 0;
        filename = 0;
        Serial.println(clientline);
        
        // Special case for the root of the site
        // if (strstr(clientline, "GET / ") != 0) {
        //  filename = rootFileName;
        // }
        
        // This is a GET request
        if (strstr(clientline, "GET /") != 0) {
          
          // Handle special case requests
          if (strstr(clientline, "GET / ") != 0) {
            printHeader(client, 0);
            client.println("Welcome to the Arduino Aquarium Controller");
            printFooter(client);
          }
          else if (strstr(clientline, "GET /analog") != 0) {
            printHeader(client, 3);
            
            for (int analogChannel = 0; analogChannel < 6; analogChannel++) {
              client.print("analog input ");
              client.print(analogChannel);
              client.print(" is ");
              client.print(analogRead(analogChannel));
              client.println("<br />");
            }
            printFooter(client); 
          }
          else if (strstr(clientline, "GET /tempinit") != 0) {
            printHeader(client, 1);
            
            client.println("Starting sensors init<br>");
            
            sensors.begin();
            
            numberOfDevices = sensors.getDeviceCount();
            client.print("Found ");
            client.print(numberOfDevices, DEC);
            client.println(" devices. <br>");
            
            // report parasite power requirements
            client.print("Parasite power is: "); 
            if (sensors.isParasitePowerMode())
              client.println("ON<br>");
            else
              client.println("OFF<br>");
              
            for(int i=0;i<numberOfDevices; i++)
            {
              // Search the wire for address
              if(sensors.getAddress(tempDeviceAddress, i))
          	{
          		client.print("Found device ");
          		client.print(i, DEC);
          		client.print(" with address: ");
          		printAddress(tempDeviceAddress, client);
          		client.println("<br>");
          		
          		client.print("Setting resolution to ");
          		client.print(TEMPERATURE_PRECISION, DEC);
                        client.println(" bits<br>");
          		
          		// set the resolution to TEMPERATURE_PRECISION bit (Each Dallas/Maxim device is capable of several different resolutions)
          		sensors.setResolution(tempDeviceAddress, TEMPERATURE_PRECISION);
          		
          		client.print("Resolution actually set to: ");
          		client.print(sensors.getResolution(tempDeviceAddress), DEC); 
          		client.println("bits <br>");
          	}else{
          		client.print("Found ghost device at ");
          		client.print(i, DEC);
          		client.print(" but could not detect address. Check power and cabling");
                        client.println("<br>");
          	}
            }  
              
            client.println("End sensors init<br>");
            printFooter(client);  
          }
          else if (strstr(clientline, "GET /temp ") != 0) {
            printHeader(client, 2);
            
            client.println("<h1>Temperature Sensors</h1>");
            
            sensors.requestTemperatures();
         
            client.println("<table>");
            client.println("<tr><th>Index</th><th>Address</th><th>Temperature</th></tr>");
            
            for(int i=0;i<numberOfDevices; i++)
            {
              client.println("<tr>");
              if(sensors.getAddress(tempDeviceAddress, i))
              {
                float tempC = sensors.getTempC(tempDeviceAddress);
                
                client.print("<td>");
                client.print(i);
                client.print("</td>");
                client.print("<td>");
                printAddress(tempDeviceAddress, client);
                client.print("</td>");
                client.print("<td>");
                client.print(tempC);
                client.print("&deg;C</td>");
              }
              else
              {
                client.print("<td>");
                client.print(i);
                client.print("</td>");
                client.print("<td>");
                printAddress(tempDeviceAddress, client);
                client.print("</td>");
                client.print("<td>");
                client.print("Ghost");
                client.print("</td>");
              }
              client.println("</tr>");
            }
            client.println("</table>");
            printFooter(client);
          }
          
          // Not a special case, try and display file from SD Card
          else {
            // If filename is not already set, get it from the GET request
            if (!filename)
              filename = clientline + 5;
            // Null terminate the string after the filename
            (strstr(clientline, " HTTP"))[0] = 0;
            
            // Add the root web directory to the filename
            strcpy (webfilename, rootWebDir);
            strcat (webfilename, "/");
            strcat (webfilename, filename);
         
           // Try and open the file from the SD Card
            Serial.println(webfilename);          
            if (! file.open(&root, webfilename, O_READ)) {
              // File does not exist
              client.println("HTTP/1.1 404 Not Found");
              client.println("Content-Type: text/html");
              client.println();
              client.println("<h2>File Not Found!</h2>");
              break;
            }
          
            // File opened
            Serial.println("Opened!");
            client.println("HTTP/1.1 200 OK");
            // Print header and content type based on file extension
            if (strstr(filename, ".htm") != 0)
              client.println("Content-Type: text/html");
            else if (strstr(filename, ".css") != 0)
              client.println("Content-Type: text/css");
            else if (strstr(filename, ".png") != 0)
              client.println("Content-Type: image/png");
            else if (strstr(filename, ".jpg") != 0)
              client.println("Content-Type: image/jpeg");
            else if (strstr(filename, ".gif") != 0)
              client.println("Content-Type: image/gif");
            else if (strstr(filename, ".3gp") != 0)
              client.println("Content-Type: video/mpeg");
            else if (strstr(filename, ".pdf") != 0)
              client.println("Content-Type: application/pdf");
            else if (strstr(filename, ".js") != 0)
              client.println("Content-Type: application/x-javascript");
            else if (strstr(filename, ".xml") != 0)
              client.println("Content-Type: application/xml");
            else
              client.println("Content-Type: text");
            client.println("Last-Modified: Sun, 4 Mar 2012 21:39:26 GMT");
            client.println();
          
            // Read the file and print the output to the client, using a buffer
            int16_t c;
            bufindex=0;  //reset buffer index
            while ((c = file.read()) >= 0) {
              buf[bufindex++]=((char)c);  //fill buffer
              if(bufindex==maxbyte)  //empty buffer when maximum length is reached.
              {
                client.write(buf, maxbyte);
                bufindex=0;
              }
            }
            file.close();  //close the file
            if(bufindex>0)  //most likely, the size of the file will not be an even multiple of the buffer length, so any remaining data is read out.
            {
              client.write(buf, bufindex);
            }
            bufindex=0; //reset buffer index (reset twice for redundancy
          
          }
        } 
        else {
          // This is not a GET request, print 404 not found
          client.println("HTTP/1.1 404 Not Found");
          client.println("Content-Type: text/html");
          client.println();
          client.println("<h2>File Not Found!</h2>");
        }
        break;
      }
    }
    delay(1);
    client.stop();
  }
}


// function to print a device address
void printAddress(DeviceAddress deviceAddress, EthernetClient client)
{
  for (uint8_t i = 0; i < 8; i++)
  {
    if (deviceAddress[i] < 16) client.print("0");
    client.print(deviceAddress[i], HEX);
  }
}


void printHeader(EthernetClient client, int menuIndex)
{
  // Print HTTP header
  client.println("HTTP/1.1 200 OK");
  client.println("Content-Type: text/html");
  client.println();
  // Print HTML header stuff - this could be moved to the SD card depending on memory usage
  client.println("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\">");
  client.println("<html xmlns=\"http://www.w3.org/1999/xhtml\">");
  client.println("<head>");
  client.println("<title>Arduino Aquarium Controller</title>");
  client.println("<link rel=\"stylesheet\" type=\"text/css\" href=\"style.css\" />");
  client.println("</head>");
  client.println("<body>");
  client.println("<div id=\"title\">Arduino Aquarium Controller</div>");
  client.println("<div class=\"menu\">");
  client.println("<ul>");
  // The menu goes here
  
  for (int i=0; i<numberOfMenuItems; i++)
  {
    client.print("<li");
    if (i == menuIndex)
      client.print(" class=\"selected\"");
    client.print("><a href=\"");
    client.print(menuItems[i][1]);
    client.print("\">");
    client.print(menuItems[i][0]);
    client.println("</a></li>");
  }
  client.println("</ul>");
  client.println("</div>");
  
  //The status bar
  client.println("<div id=\"status\">");
  client.println("<span class=\"green\"><strong>Green Status</strong></span>");
  client.println("<span class=\"red\"><strong>Red Status</strong></span>");
  client.println("<span class=\"orange\"><strong>Orange Status</strong></span>");
  client.println("<div id=\"version\">Author: Andre Miller  Version: 2012-03-08</div>");
  client.println("</div>");

}

void printFooter(EthernetClient client)
{
  client.println("</body>");
  client.println("</html>");
}


